/***********************  Bibliotheken  *****************************/
#include <Wire.h>
#include <LiquidCrystal.h>

/***********************  Definitionen  *****************************/
// LCD Pin-Definitionen
//Angabe der Pins (RS, EN, D4, D5, D6, D7) des angeschlossenen LCD
#define RS 12
#define EN 11
#define D4 5
#define D5 4
#define D6 3
#define D7 2


#define expander B00111111  // Adresse des PCF8574A mit allen 3 Adress-Pins nach +5V, B00100111 waere fuer den PCF8574 (siehe Datenblatt)
#define LM75A_01 0x48       // Basisadresse fuer ersten Temperatursensor

// Registerparameter fuer get_LM75_temperature
#define TEMP 0            // Temperaturregister anwählen
#define UNTEN 2           // Register fuer den unteren Schaltwert anwählen
#define OBEN 3            // Register fuer den oberen Schaltwert anwählen

//Temperaturgrenzen LM75A
#define Wert_OBEN 32.0
#define Wert_UNTEN 27.0

//LED Ampel-Muster
//Ampel 1 (P0-P2)
#define rot1 B00000001
#define gelb1 B00000010
#define gruen1 B00000100

//Ampel 2 (P3-P5)
#define rot2 B00001000
#define gelb2 B00010000
#define gruen2 B00100000

//Taster
#define Taster_Heiz_AN 6
#define Taster_Heiz_AUS 7

//Heizungen
#define Heiz1 A1
#define Heiz2 A2
#define Heiz3 A3

//Luefter
#define Luefter1 10
#define Luefter2 13

//GasSensor
#define GasIn A0
#define R0 0.27
#define Gasstufe_1 15
#define Gasstufe_2 20
#define Gasstufe_3 50
#define Gasstufe_4 65

//WarnTonAusgänge
#define Warn1 8
#define Warn2 9

/**************GLOBALE VARIABLEN*************************/
char dataString[6];                       // gelesene Temperatur als String aufbereitet: (-xx)x.x
double temp;                              // gelesene Temperatur als double
double temp_act[6];                       // gelesene Temperatur als double
int gas_konz;                           // Variable fuer die GasKonzentration
int gas_konz_alt = 0;
unsigned long previousMillis = 0;         // Variable für Zeitberechung mit millis() 
unsigned long previousMillis10s = 0;      // Variable für Zeitberechung mit millis() 10 Sekunden
unsigned long currentMillis =  0;
unsigned long currentMillis10s =  0;
unsigned long previousMillis5s = 0;      // Variable für Zeitberechung mit millis() 5 Sekunden
unsigned long currentMillis5s =  0;
const long interval_gruen = 5000;         // Wartezeit bei grün, bevor wieder auf gelb geschaltet wird (5 Sekunden)
const long interval_wechsel = 1000;       // Wartezeit bei Einspurverkehr, bevor weiter geschaltet wird (1 Sekunde)
const long interval_10s = 10000;          // Wartezeit 10 Sekunden
int ampel_case = 1;                       // Variabel zum Speichern des aktuellen Zustands der Ampeln
int luefter_case = 0;                     // Variabel zum Speichern des aktuellen Zustands der Lüfter
int temp_dir = 1;                         // Flagge zur Temperaturrichtungserkennung: 1 steht für Temp von niedrig nach hoch; 2 steht für Temp von hoch nach niedrig
int HeizCount = 0;                        // Zählvariable zum Durchzählen der Heizausgänge
int HeizOff = 1;                          // Flagge zum Abschalten aller Heizausgänge

  
/*****************************************************/
/***************  UNTERPROGRAMME  ********************/
/*****************************************************/

/****************  LCD Init  ************************/
LiquidCrystal lcd(RS, EN, D4, D5, D6, D7);

/****************  PCF8574   ************************/
//Daten auf PCF8574A schreiben
void DataWrite(byte data)
{
  data = ~data;                                 //Daten einmal negieren, da LED mit invertierter Logik eingeschaltet werden  
  Wire.beginTransmission(expander);             //Öffnen der Verbindung
  Wire.write(data);                             //Schreiben des Bytes. Jede Null stellt ein Bit an den Pins des PCF8574 dar (Auch möglich in Hex Schreibweise)
  Wire.endTransmission();                       //Schließen der Verbindung
}


/****************  LM75  ***************************/
// LM75 Temperatur auslesen. Device = 0-7, regx = TEMP, OBEN, UNTEN (Registerauswahl)  
double get_LM75_temperature(int device, int regx) 
{
  int8_t msb;
  int8_t lsb;
  int8_t msb1;
  Wire.beginTransmission(LM75A_01 + device);   // Verbindung zum entsprechenden Temp-Sensor
  Wire.write(regx);                            // Gewünschtes Register vom Temp-Sensor ansprechen
  Wire.endTransmission();
  Wire.beginTransmission(LM75A_01 + device);
  Wire.requestFrom(LM75A_01 + device, 2);
  if (Wire.available()) {
     msb1 = Wire.read();                      // Aktuellen Temperaturwert auslesen
     msb = msb1 << 1;                         // Vorzeichenbit entfernen, verbliebener Wert ist nun doppelt so groß
     lsb = Wire.read();                       // Temp.-Wert nochmal einlesen für spätere Berechnung
  }
  // höchstes bit von lsb sagt aus, ob 0,5 Grad dazu addiert werden sollen
  lsb = (lsb & 0x80 ) >> 7;                   // nun ist lsb = 0 oder 1
  Wire.endTransmission();
  if (msb1 < 0x80) {                          // Positiver Wert?
    return double(msb + lsb)/2;               // positiver Wert
  }  
  else {
    return double(msb + lsb)/2 - 128;         // negativer Wert
  }  
}

// LM75 Konfigurationsregister setzen, Werte wie oben definiert
void set_LM75_config(int device, byte value) 
{
  Wire.beginTransmission(LM75A_01 + device);
  Wire.write(1);                                // Konfigurationsregister auswählen
  Wire.write(value);                            
  Wire.endTransmission();
}

// LM75 Konfigurationsregister auslesen, device = 0-7
byte get_LM75_config(int device) 
{
  byte reg;
  Wire.beginTransmission(LM75A_01 + device);
  Wire.write(1);                                // Konfigurationsregister auswählen
  Wire.endTransmission();
  Wire.requestFrom(LM75A_01 + device, 1);
  if (Wire.available()) { 
     reg = Wire.read();
  }
  Wire.endTransmission();
  return reg;
} 

// LM75 Schaltwerte setzen, device = 0-7, regx = Wert, Grad als double
//gleiches Vorgehen, wie beim Auslesen der Temperaturwerte, nur, dass hier geschrieben wird
void set_LM75_schaltwert(int device, byte regx, double grad) 
{
  int8_t msb;
  int8_t lsb = 0;
  uint8_t y = 0;
  boolean neg = false;
  if (grad < 0) {
    msb = abs(int(grad))+128;
  }
  else {  
    msb = abs(int(grad));
  }
  if (grad - abs(int(grad)) > 0) {
    lsb = 0x80;
  }
  Wire.beginTransmission(LM75A_01 + device);
  Wire.write(regx);                                   // Selektiere oberes oder unteres Register
  Wire.write(msb);
  Wire.write(lsb);
  Wire.endTransmission();
}

// Temperaturen aller 6 Sensoren einlesen und auf LCD ausgeben
int readTemp6()
{
  // Temperatur von allen 6 LM75 auslesen
  int temp_case = 0;                                      // Flagge fuer Temperaturwertueberschreitung
  for (int i=0;i<6;i++)
  {
    temp = get_LM75_temperature(i, TEMP);                 //(Device)Wert vom 1. Temperatursensor lesen (0-7, je nach Jumperstellung am Board, 2. Parameter wie oben definiert)
    temp_act[i] = temp;

    if (temp_act[i] >= Wert_OBEN)                         //Wenn die gemessene Temperatur größer als beide Grenzen ist
    {
      temp_case = 2;
    }
    else if((temp_act[i] > Wert_UNTEN && temp_act[i] < Wert_OBEN) && temp_case != 2 && temp_dir == 2)    //Wenn die gemessene Temperatur größer als untere Grenzen. aber kleiner als obere Grenze ist und keine anderer Sensor ueber der oberen Grenze liegt
    {
      temp_case = 1;
    }

    dtostrf(temp, 4, 1, dataString);                       //dtostrf(floatVar, minStringWidthIncDecimalPoint, numVarsAfterDecimal, charBuf); standard avr-libc function ; damit Tempwert auf eine Nachkommastelle abgeschnitten wird
    
    //LCD Ausgabe Temp
    switch (i+1)                                          // Positionen auf LCD werden je nach aktuellem Temp-Sensor entpsprechend versetzt
    {
      case 1:
        lcd.setCursor(0, 0);
        break;
      case 2:
        lcd.setCursor(9, 0);
        break;
      case 3:
        lcd.setCursor(0, 1);
        break;
      case 4:
        lcd.setCursor(9, 1);
        break;
      case 5:
        lcd.setCursor(0, 2);
        break;
      case 6:
        lcd.setCursor(9, 2);
        break;
      default:
        break;
    }  
    lcd.print("T");
    lcd.print(i+1);
    lcd.print(":");
    lcd.print(dataString);
  }
  if(temp_case == 2)                                         // Ist der Temp-Case 2 erreicht (Obere Grenze wurde überschritten) wird temp_dir (Temperaturrichtung) auf 2 gestellt, um zu signalisieren, dass das nächste mal im Bereich zwischen OBEN und UNTEN in Richtung unten erreicht wird
    temp_dir = 2;
  return temp_case;
}



/************************* GasSensor - MQ3 *************************/
float MQ3() 
{
   // Original Routine vom Herstelle von MQ3 Sensor für die Kalibirieung des Sensorsignals
  /*-Replace the name "R0" with the value of R0 in the demo of First Test -*/
  /* R0 ist unter den Definitionen zu finden */
  float sensor_volt;
  float RS_gas;             // Get value of RS in a GAS
  float ratio;              // Get ratio RS_GAS/RS_air
  int sensorValue = analogRead(A0);
  sensor_volt=(float)sensorValue/1024*5.0;
  RS_gas = (5.0-sensor_volt)/sensor_volt; // omit *RL
  ratio = RS_gas/R0;      // ratio = RS/R0

  return ratio;
}

int readGas()
{
  int gas_Case = 0;
  //Gaswert auslesen
//  gas_konz = MQ3();
  gas_konz = int(map(analogRead(A0),0,1023,0,100));
  if (gas_konz != gas_konz_alt)           // um Flackern der Anzeige zu vermeiden soll der Gas-Wert nur dann neu geschrieben werden, falls er sich zum letzten hin geändert hat
  {
    gas_konz_alt = gas_konz;
    lcd.setCursor(0, 3);
    lcd.print("                ");        // Der Anzeigebereich wird jedes mal mit Leerzeichen überschrieben, damit alte Daten der Gas-Konzentration (bsp: Sprung von 33% auf 7%, oder ALRAM) keine Fragmente hintelassen
    lcd.setCursor(0, 3);
    lcd.print("Gas:");
    lcd.print(gas_konz);
    lcd.print(" %");

  }
  if(gas_konz >= Gasstufe_2)      //zweiter Gas Grenzwert ueberschritten
  {
    lcd.print("      ");
    gas_Case = 2;
  }
    if(gas_konz >= Gasstufe_4)      //oberer Gas Grenzwert ueberschritten
  {
    lcd.setCursor(10, 3);
    lcd.print(" ALARM");
    gas_Case = 4;
  }
  else
  {
    lcd.print("      ");
  }
  return gas_Case;
}


/******************* Lüfter- und Warnton-Steuerung  ****************/
void Luefter(int temp_case, int Gas_case)
{
  //beide Lüfter und Warntöne aus
  if(temp_case == 0 && Gas_case == 0)
  {
    luefter_case = 0;
  }
  //Lüfter 1 und Warnton 1 an
  if(temp_case == 0 && (Gas_case == 2))
  {
    luefter_case = 1;
  }
  //Lüfter 1&2 und Warnton 1&2 an
  if((temp_case == 0 || temp_case == 2) && Gas_case == 4)
  {
    luefter_case = 2;
  }
  //Lüfter 2 und Warnton 2 an, Lüfter 1 noch 5 Sekunden Nachlaufzeit
  if(temp_case == 2 && (Gas_case == 0 || Gas_case == 1))
  {
    luefter_case = 3;
  } 
  //Lüfter 1&2 und Warnton 1&2 an
  if(temp_case == 2 && (Gas_case == 2 || Gas_case == 4))
  {
    luefter_case = 4;

  }
     
  currentMillis5s =  millis();
  switch (luefter_case)                         // Fallunterscheidung für die einzelnen Lüfterphasen 
  {                                           
    case 0:                                   //Alle Lüfter aus
        if (currentMillis5s - previousMillis5s >= interval_gruen)     //Intervall_gruen = 5 Sekunden
        {
           digitalWrite(Luefter1, LOW);         
        }
        digitalWrite(Luefter2, LOW);
        digitalWrite(Warn1, LOW);
        digitalWrite(Warn2, LOW);
        break;
    case 1:                                   
        digitalWrite(Luefter1, HIGH);          //Lüfter 1 einschalten
        digitalWrite(Luefter2, LOW);         //Lüfter 2 ausschalten
        digitalWrite(Warn1, LOW);             //WarnTon1 einschalten
        digitalWrite(Warn2, LOW);            //WarnTon2 ausschalten
        previousMillis5s = currentMillis5s;
        break;
    case 2:
        digitalWrite(Luefter1, HIGH);          //Lüfter 1 einschalten
        digitalWrite(Luefter2, HIGH);          //Lüfter 2 einschalten
        digitalWrite(Warn1, HIGH);             //WarnTon1 einschalten
        digitalWrite(Warn2, HIGH);             //WarnTon2 einschalten
        previousMillis5s = currentMillis5s;
        break;
    case 3:
        if (currentMillis5s - previousMillis5s >= interval_gruen)     //Intervall_gruen = 5 Sekunden
        {
          digitalWrite(Luefter1, LOW);         //Lüfter 1 ausschalten
        }
        digitalWrite(Luefter2, HIGH);          //Lüfter 2 einschalten
        digitalWrite(Warn1, LOW);            //WarnTon1 ausschalten
        digitalWrite(Warn2, HIGH);             //WarnTon2 einschalten
        break;
     case 4:
        digitalWrite(Luefter1, HIGH);          //Lüfter 1 einschalten
        digitalWrite(Luefter2, HIGH);          //Lüfter 2 einschalten
        digitalWrite(Warn1, HIGH);             //WarnTon1 einschalten
        digitalWrite(Warn2, HIGH);             //WarnTon2 einschalten
        previousMillis5s = currentMillis5s;
        break;
      default:
        previousMillis5s = currentMillis5s;
        break;
             
  }
}


/*************************  Ampel-Steuerung  ***********************/

void Ampel(int temp_case, int Gas_case)
{

  if(Gas_case == 4)                               //Temperatur ueber oberem Grenzwert
  {
    if(ampel_case == 5 || ampel_case == 10)                         //es wird noch so lange gewartet, bis beide Ampeln durch Einspurverhalten auf Rot schalten und verbleiben dann dort
    {
      ampel_case = 9;
      Ampel_strg();
    }
    else
      {
        Ampel_strg();
      }
  }
  else if((temp_case == 0 || temp_case == 1 || temp_case == 2) && Gas_case == 2)                          //Temperatur ist ueber oberem Grenzwert oder Gas über unterem Wert
  {
    //Ampel einspurig
    Ampel_strg();                                                   //Ampel läuft ab aktuellem Stand in das Einspurverhalten
    previousMillis10s = currentMillis10s;
  }
  else if((temp_case == 1 || temp_case == 2) && (Gas_case == 0))                          //Temperatur ist ueber oberem Grenzwert und Gas ist i.O.
  {
    //Ampel einspurig
    Ampel_strg();                                                   //Ampel läuft ab aktuellem Stand in das Einspurverhalten
    previousMillis10s = currentMillis10s;
  }
  else if(temp_case == 0 && (Gas_case == 0 || Gas_case == 1))                          //Temperatur liegt unter unterem Grenzwert
  {
    currentMillis10s = millis();
    if (currentMillis10s - previousMillis10s >= interval_wechsel) 
    {
     // previousMillis10s = currentMillis10s;
    
      if(ampel_case == 11)
      {
        ampel_case = 12;
        Ampel_strg();               //Beide Ampeln werden auf gruen geschaltet
      }
      else if (ampel_case == 7)
      {
        ampel_case = 14;
        Ampel_strg();               //Beide Ampeln werden auf gruen geschaltet
      }
      else if (ampel_case == 1)
      {
        Ampel_strg();               //Beide Ampeln werden auf gruen geschaltet
        ampel_case = 1;
      }
      else
      {
        Ampel_strg();
      }
    }
  }
}

void Ampel_strg()
{
  switch (ampel_case)                         // Fallunterscheidung für die einzelnen Ampelphasen (s. Doku)
  {                                           // bei jedem Aufruf der Funktion Ampel_strg() wird der gleihe case eingenommen bis 1 Sekunde 
    case 1:                                   // vergangen ist. Dann wird der auf den nächsten case umgeschaltet, welcher das nächste Ampel-
        DataWrite(gruen1 | gruen2);           // muster beinhaltet
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 2;
        }
        break;
    case 2:
        DataWrite(gelb1 | gruen2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 3;
        }
        break;
    case 3:
        DataWrite(rot1 | gruen2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_gruen) 
        {
          previousMillis = currentMillis;
          ampel_case = 4;
        }
        break;
    case 4:
        DataWrite(rot1 | gelb2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 5;
        }
        break;
    case 5:
        DataWrite(rot1 | rot2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 6;
        }
        break;
    case 6:
        DataWrite((rot1 | gelb1) | rot2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 7;
        }
        break;
    case 7:
        DataWrite(gruen1 | rot2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_gruen)   // Das Intervall für die Pause bei Grün ist 5 Sekunden lang
        {
          previousMillis = currentMillis;
          ampel_case = 8;
        }
        break;
    case 8:
        DataWrite(gelb1 | rot2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 9;
        }
        break;
    case 9:
        DataWrite(rot1 | rot2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 10;
        }
        break;
    case 10:
        DataWrite(rot1 | (rot2 | gelb2));
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 11;
        }
        break;
    case 11:
        DataWrite(rot1 | gruen2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_gruen) 
        {
          previousMillis = currentMillis;
          ampel_case = 4;                             // um den Kreislauf der Ampelphasen herzustellen wird nach case 11 wieder auf case 4 
        }                                             // gesprungen
        break;
    case 12:
        DataWrite((rot1 | gelb1) | gruen2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 13;
        }
        break;
    case 13:
        DataWrite(gruen1 | gruen2);
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel) 
        {
          previousMillis = currentMillis;
          ampel_case = 1;
        }
        break;
    case 14:
        DataWrite(gruen1 | (rot2 | gelb2));
        currentMillis = millis();
        if (currentMillis - previousMillis >= interval_wechsel)   
        {
          previousMillis = currentMillis;
          ampel_case = 13;
        }
        break;
    default:
        break;
  }
}

/*************************  Heizungs-Steuerung  ***********************/

void HeizStrg()
{
  if(digitalRead(Taster_Heiz_AN) == LOW)
  {
    delay(10);                                                // Software-Entprellung des Tasters
    if(digitalRead(Taster_Heiz_AN) == LOW)
    {
      HeizCount++;
      if(HeizCount >= 5)
      {
        HeizCount = 1;
      }
      HeizOff = 0;
    }
  }
  if(digitalRead(Taster_Heiz_AUS) == LOW)
  {
    delay(10);                                                // Software-Entprellung des Tasters
    if(digitalRead(Taster_Heiz_AUS) == LOW)
    {
      HeizCount = 1;                                          // Falls der Aus-Taster gedrückt wird, werden alle Heizungen abgeschaltet und der HeizCounter wieder auf 1 gesetzt 
      HeizOff = 1;
    }
  }
  if(HeizOff == 1)
  {
    digitalWrite(Heiz1, LOW);
    digitalWrite(Heiz2, LOW);
    digitalWrite(Heiz3, LOW);
  }
  else
  {
    switch(HeizCount)
    {
      case 1:
          digitalWrite(Heiz1, HIGH);
          digitalWrite(Heiz2, LOW);
          digitalWrite(Heiz3, LOW);
          break;
      case 2:
          digitalWrite(Heiz1, LOW);
          digitalWrite(Heiz2, HIGH);
          digitalWrite(Heiz3, LOW);
          break;
      case 3:
          digitalWrite(Heiz1, LOW);
          digitalWrite(Heiz2, LOW);
          digitalWrite(Heiz3, HIGH);
          break;
      case 4:
          digitalWrite(Heiz1, HIGH);
          digitalWrite(Heiz2, HIGH);
          digitalWrite(Heiz3, HIGH);
          break;
      
      default:
          digitalWrite(Heiz1, LOW);
          digitalWrite(Heiz2, LOW);
          digitalWrite(Heiz3, LOW);
          break;
    }
  }
  
}

/*************************  SETUP  *********************************/

void setup() {
  Wire.begin();                                               //Initialisieren der I2C Verbindung. MUSS im Setup aufgerufen werden.
  Serial.begin(9600);
  lcd.begin(16, 4);                                           // Einstellung des LCD auf 16 Spalten auf 4 Zeilen
  pinMode(Taster_Heiz_AN, INPUT);
  pinMode(Taster_Heiz_AUS, INPUT);
  pinMode(GasIn, INPUT);
  pinMode(Warn1, OUTPUT);
  pinMode(Warn2, OUTPUT);
  pinMode(Heiz1, OUTPUT);
  pinMode(Heiz2, OUTPUT);
  pinMode(Heiz3, OUTPUT);
  pinMode(Luefter1, OUTPUT);
  pinMode(Luefter2, OUTPUT);

  digitalWrite(Warn1, HIGH);
  digitalWrite(Warn2, HIGH);
  digitalWrite(Heiz1, LOW);
  digitalWrite(Heiz2, LOW);
  digitalWrite(Heiz3, LOW);
  digitalWrite(Luefter1, HIGH);
  digitalWrite(Luefter2, HIGH);
  
  for (int i=0;i<6;i++)                           // alle 6 LM75 Sensoren konfigurieren
  {
    set_LM75_config(i, 0);                        // LM75 Konfigurationsregister setzen: Device, Wert siehe oben
    set_LM75_schaltwert(i, UNTEN, Wert_UNTEN);    // LM75 Schaltwerte setzen: Device, Register, Wert als double 
    set_LM75_schaltwert(i, OBEN, Wert_OBEN);
  }
  lcd.setCursor(4, 0);                            // LCD zu Beginn mit Prüfungsname bespielen und Ausgabe auf serieller Schnittstelle
  lcd.print("Pruefung ");                      // zur Überprüfung beim flashen und 5 Sekunden warten 
  lcd.setCursor(2, 1);
  lcd.print("Winter 19/20");
  lcd.setCursor(3, 2);
  lcd.print("Simulation");
  lcd.setCursor(2, 3);
  lcd.print("Tunnelanlage");

  Serial.println("Pruefung");
  Serial.println("Winter 19/20");
  Serial.println("Simulation");
  Serial.println("Tunnelanlage");
  
  delay(5000);
  lcd.clear(); 
  
  gas_konz = map(analogRead(A0),0,1023,0,100);
  lcd.setCursor(0, 3);
  lcd.print("Gas:");
  lcd.print(gas_konz);
  lcd.print("%");
 
 
}

/*********************** HAUPTPROGRAMM  ***************************/

void loop()
{
  int gas_case = readGas();                         // Gas-Wert auslesen und auf LCD schreiben
  int temp_case = readTemp6();                      // Flagge fuer Temperaturwertueberschreitung
  Ampel(temp_case,gas_case);                        // Ampel-Funktion aufrufen nachdem Gas- und Temperaturwerte abgeholt wurden  
  Luefter(temp_case,gas_case);                      // Lüfter-Funktion aufrufen nachdem Gas- und Temperaturwerte abgeholt wurden
  HeizStrg();                                       // Heizungs-Steuerung aufrufen, um die Heizwiderstände an den LM75 zu steuern
  delay(250);
}
